/*
 * Decompiled with CFR 0.152.
 */
package com.gitblit.manager;

import com.gitblit.Constants;
import com.gitblit.GitBlitException;
import com.gitblit.IStoredSettings;
import com.gitblit.extensions.RepositoryLifeCycleListener;
import com.gitblit.manager.IPluginManager;
import com.gitblit.manager.IRepositoryManager;
import com.gitblit.manager.IRuntimeManager;
import com.gitblit.manager.IUserManager;
import com.gitblit.models.ForkModel;
import com.gitblit.models.Metric;
import com.gitblit.models.RefModel;
import com.gitblit.models.RegistrantAccessPermission;
import com.gitblit.models.RepositoryCommit;
import com.gitblit.models.RepositoryModel;
import com.gitblit.models.SearchResult;
import com.gitblit.models.TeamModel;
import com.gitblit.models.UserModel;
import com.gitblit.service.GarbageCollectorService;
import com.gitblit.service.LuceneService;
import com.gitblit.service.MirrorService;
import com.gitblit.utils.ArrayUtils;
import com.gitblit.utils.ByteFormat;
import com.gitblit.utils.CommitCache;
import com.gitblit.utils.DeepCopier;
import com.gitblit.utils.JGitUtils;
import com.gitblit.utils.MetricUtils;
import com.gitblit.utils.ModelUtils;
import com.gitblit.utils.ObjectCache;
import com.gitblit.utils.StringUtils;
import com.gitblit.utils.TimeUtils;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryCache;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.storage.file.FileBasedConfig;
import org.eclipse.jgit.storage.file.WindowCacheConfig;
import org.eclipse.jgit.util.FS;
import org.eclipse.jgit.util.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class RepositoryManager
implements IRepositoryManager {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final ScheduledExecutorService scheduledExecutor = Executors.newScheduledThreadPool(5);
    private final ObjectCache<Long> repositorySizeCache = new ObjectCache();
    private final ObjectCache<List<Metric>> repositoryMetricsCache = new ObjectCache();
    private final Map<String, RepositoryModel> repositoryListCache = new ConcurrentHashMap<String, RepositoryModel>();
    private final AtomicReference<String> repositoryListSettingsChecksum = new AtomicReference<String>("");
    private final IStoredSettings settings;
    private final IRuntimeManager runtimeManager;
    private final IPluginManager pluginManager;
    private final IUserManager userManager;
    private File repositoriesFolder;
    private LuceneService luceneExecutor;
    private GarbageCollectorService gcExecutor;
    private MirrorService mirrorExecutor;

    @Inject
    public RepositoryManager(IRuntimeManager runtimeManager, IPluginManager pluginManager, IUserManager userManager) {
        this.settings = runtimeManager.getSettings();
        this.runtimeManager = runtimeManager;
        this.pluginManager = pluginManager;
        this.userManager = userManager;
    }

    @Override
    public RepositoryManager start() {
        this.repositoriesFolder = this.runtimeManager.getFileOrFolder("git.repositoriesFolder", "${baseFolder}/git");
        this.logger.info("Repositories folder : {}", (Object)this.repositoriesFolder.getAbsolutePath());
        String prefix = this.settings.getString("git.userRepositoryPrefix", "~");
        ModelUtils.setUserRepoPrefix(prefix);
        this.repositoryListSettingsChecksum.set(this.getRepositoryListSettingsChecksum());
        if (this.settings.getBoolean("git.cacheRepositoryList", true)) {
            this.logger.info("Identifying repositories...");
            this.getRepositoryList();
        }
        this.configureLuceneIndexing();
        this.configureGarbageCollector();
        this.configureMirrorExecutor();
        this.configureJGit();
        this.configureCommitCache();
        this.confirmWriteAccess();
        return this;
    }

    @Override
    public RepositoryManager stop() {
        this.scheduledExecutor.shutdownNow();
        this.luceneExecutor.close();
        this.gcExecutor.close();
        this.mirrorExecutor.close();
        this.closeAll();
        return this;
    }

    @Override
    public Date getLastActivityDate() {
        Date date = null;
        for (String name : this.getRepositoryList()) {
            Repository r = this.getRepository(name);
            Date lastChange = JGitUtils.getLastChange((Repository)r).when;
            r.close();
            if (lastChange == null || date != null && !lastChange.after(date)) continue;
            date = lastChange;
        }
        return date;
    }

    @Override
    public File getRepositoriesFolder() {
        return this.repositoriesFolder;
    }

    @Override
    public File getHooksFolder() {
        return this.runtimeManager.getFileOrFolder("groovy.scriptsFolder", "${baseFolder}/groovy");
    }

    @Override
    public File getGrapesFolder() {
        return this.runtimeManager.getFileOrFolder("groovy.grapeFolder", "${baseFolder}/groovy/grape");
    }

    @Override
    public boolean isCollectingGarbage() {
        return this.gcExecutor != null && this.gcExecutor.isRunning();
    }

    @Override
    public boolean isCollectingGarbage(String repositoryName) {
        return this.gcExecutor != null && this.gcExecutor.isCollectingGarbage(repositoryName);
    }

    @Override
    public List<RegistrantAccessPermission> getUserAccessPermissions(UserModel user) {
        if (StringUtils.isEmpty(user.username)) {
            return new ArrayList<RegistrantAccessPermission>();
        }
        LinkedHashSet<RegistrantAccessPermission> set = new LinkedHashSet<RegistrantAccessPermission>();
        set.addAll(user.getRepositoryPermissions());
        for (RegistrantAccessPermission permission : set) {
            RepositoryModel rm;
            if (!permission.mutable || !Constants.PermissionType.EXPLICIT.equals((Object)permission.permissionType) || (rm = this.getRepositoryModel(permission.registrant)) != null) continue;
            permission.permissionType = Constants.PermissionType.MISSING;
            permission.mutable = false;
        }
        for (RepositoryModel rm : this.repositoryListCache.values()) {
            if (!rm.isUsersPersonalRepository(user.username) && !rm.isOwner(user.username)) continue;
            RegistrantAccessPermission rp = new RegistrantAccessPermission(rm.name, Constants.AccessPermission.REWIND, Constants.PermissionType.OWNER, Constants.RegistrantType.REPOSITORY, null, false);
            set.remove(rp);
            set.add(rp);
        }
        ArrayList<RegistrantAccessPermission> list = new ArrayList<RegistrantAccessPermission>(set);
        Collections.sort(list);
        return list;
    }

    @Override
    public List<RegistrantAccessPermission> getUserAccessPermissions(RepositoryModel repository) {
        ArrayList<RegistrantAccessPermission> list = new ArrayList<RegistrantAccessPermission>();
        if (Constants.AccessRestrictionType.NONE.equals((Object)repository.accessRestriction)) {
            return list;
        }
        if (Constants.AuthorizationControl.AUTHENTICATED.equals((Object)repository.authorizationControl)) {
            return list;
        }
        for (UserModel user : this.userManager.getAllUsers()) {
            RegistrantAccessPermission ap = user.getRepositoryPermission(repository);
            if (!ap.permission.exceeds(Constants.AccessPermission.NONE)) continue;
            list.add(ap);
        }
        return list;
    }

    @Override
    public boolean setUserAccessPermissions(RepositoryModel repository, Collection<RegistrantAccessPermission> permissions) {
        ArrayList<UserModel> users = new ArrayList<UserModel>();
        for (RegistrantAccessPermission up : permissions) {
            if (!up.mutable) continue;
            UserModel user = this.userManager.getUserModel(up.registrant);
            user.setRepositoryPermission(repository.name, up.permission);
            users.add(user);
        }
        return this.userManager.updateUserModels(users);
    }

    @Override
    public List<String> getRepositoryUsers(RepositoryModel repository) {
        return this.userManager.getUsernamesForRepositoryRole(repository.name);
    }

    @Override
    public List<RegistrantAccessPermission> getTeamAccessPermissions(RepositoryModel repository) {
        ArrayList<RegistrantAccessPermission> list = new ArrayList<RegistrantAccessPermission>();
        for (TeamModel team : this.userManager.getAllTeams()) {
            RegistrantAccessPermission ap = team.getRepositoryPermission(repository);
            if (!ap.permission.exceeds(Constants.AccessPermission.NONE)) continue;
            list.add(ap);
        }
        Collections.sort(list);
        return list;
    }

    @Override
    public boolean setTeamAccessPermissions(RepositoryModel repository, Collection<RegistrantAccessPermission> permissions) {
        ArrayList<TeamModel> teams = new ArrayList<TeamModel>();
        for (RegistrantAccessPermission tp : permissions) {
            if (!tp.mutable) continue;
            TeamModel team = this.userManager.getTeamModel(tp.registrant);
            team.setRepositoryPermission(repository.name, tp.permission);
            teams.add(team);
        }
        return this.userManager.updateTeamModels(teams);
    }

    @Override
    public List<String> getRepositoryTeams(RepositoryModel repository) {
        return this.userManager.getTeamNamesForRepositoryRole(repository.name);
    }

    @Override
    public void addToCachedRepositoryList(RepositoryModel model) {
        if (this.settings.getBoolean("git.cacheRepositoryList", true)) {
            String originKey;
            String key = this.getRepositoryKey(model.name);
            this.repositoryListCache.put(key, model);
            if (!StringUtils.isEmpty(model.originRepository) && this.repositoryListCache.containsKey(originKey = this.getRepositoryKey(model.originRepository))) {
                RepositoryModel origin = this.repositoryListCache.get(originKey);
                origin.addFork(model.name);
            }
        }
    }

    private RepositoryModel removeFromCachedRepositoryList(String name) {
        if (StringUtils.isEmpty(name)) {
            return null;
        }
        String key = this.getRepositoryKey(name);
        return this.repositoryListCache.remove(key);
    }

    private void clearRepositoryMetadataCache(String repositoryName) {
        this.repositorySizeCache.remove(repositoryName);
        this.repositoryMetricsCache.remove(repositoryName);
        CommitCache.instance().clear(repositoryName);
    }

    @Override
    public void resetRepositoryCache(String repositoryName) {
        this.removeFromCachedRepositoryList(repositoryName);
        this.clearRepositoryMetadataCache(repositoryName);
        this.getRepositoryModel(repositoryName);
    }

    @Override
    public void resetRepositoryListCache() {
        this.logger.info("Repository cache manually reset");
        this.repositoryListCache.clear();
        this.repositorySizeCache.clear();
        this.repositoryMetricsCache.clear();
        CommitCache.instance().clear();
    }

    private String getRepositoryListSettingsChecksum() {
        StringBuilder ns = new StringBuilder();
        ns.append(this.settings.getString("git.cacheRepositoryList", "")).append('\n');
        ns.append(this.settings.getString("git.onlyAccessBareRepositories", "")).append('\n');
        ns.append(this.settings.getString("git.searchRepositoriesSubfolders", "")).append('\n');
        ns.append(this.settings.getString("git.searchRecursionDepth", "")).append('\n');
        ns.append(this.settings.getString("git.searchExclusions", "")).append('\n');
        String checksum = StringUtils.getSHA1(ns.toString());
        return checksum;
    }

    private boolean isValidRepositoryList() {
        String newChecksum = this.getRepositoryListSettingsChecksum();
        boolean valid = newChecksum.equals(this.repositoryListSettingsChecksum.get());
        this.repositoryListSettingsChecksum.set(newChecksum);
        if (!valid && this.settings.getBoolean("git.cacheRepositoryList", true)) {
            this.logger.info("Repository list settings have changed. Clearing repository list cache.");
            this.repositoryListCache.clear();
        }
        return valid;
    }

    @Override
    public List<String> getRepositoryList() {
        if (this.repositoryListCache.size() == 0 || !this.isValidRepositoryList()) {
            long startTime = System.currentTimeMillis();
            List<String> repositories = JGitUtils.getRepositoryList(this.repositoriesFolder, this.settings.getBoolean("git.onlyAccessBareRepositories", false), this.settings.getBoolean("git.searchRepositoriesSubfolders", true), this.settings.getInteger("git.searchRecursionDepth", -1), this.settings.getStrings("git.searchExclusions"));
            if (!this.settings.getBoolean("git.cacheRepositoryList", true)) {
                StringUtils.sortRepositorynames(repositories);
                return repositories;
            }
            String msg = "{0} repositories identified in {1} msecs";
            if (this.settings.getBoolean("web.showRepositorySizes", true)) {
                msg = "{0} repositories identified with calculated folder sizes in {1} msecs";
            }
            for (String repository : repositories) {
                this.getRepositoryModel(repository);
            }
            for (RepositoryModel model : this.repositoryListCache.values()) {
                String originKey;
                if (StringUtils.isEmpty(model.originRepository) || !this.repositoryListCache.containsKey(originKey = this.getRepositoryKey(model.originRepository))) continue;
                RepositoryModel origin = this.repositoryListCache.get(originKey);
                origin.addFork(model.name);
            }
            long duration = System.currentTimeMillis() - startTime;
            this.logger.info(MessageFormat.format(msg, this.repositoryListCache.size(), duration));
        }
        ArrayList<String> list = new ArrayList<String>();
        for (RepositoryModel model : this.repositoryListCache.values()) {
            list.add(model.name);
        }
        StringUtils.sortRepositorynames(list);
        return list;
    }

    @Override
    public Repository getRepository(String repositoryName) {
        return this.getRepository(repositoryName, true);
    }

    @Override
    public Repository getRepository(String name, boolean logError) {
        Repository r;
        block4: {
            String repositoryName = this.fixRepositoryName(name);
            if (this.isCollectingGarbage(repositoryName)) {
                this.logger.warn(MessageFormat.format("Rejecting request for {0}, busy collecting garbage!", repositoryName));
                return null;
            }
            File dir = RepositoryCache.FileKey.resolve((File)new File(this.repositoriesFolder, repositoryName), (FS)FS.DETECTED);
            if (dir == null) {
                return null;
            }
            r = null;
            try {
                RepositoryCache.FileKey key = RepositoryCache.FileKey.exact((File)dir, (FS)FS.DETECTED);
                r = RepositoryCache.open((RepositoryCache.Key)key, (boolean)true);
            }
            catch (IOException e) {
                if (!logError) break block4;
                this.logger.error("GitBlit.getRepository(String) failed to find " + new File(this.repositoriesFolder, repositoryName).getAbsolutePath());
            }
        }
        return r;
    }

    @Override
    public List<RepositoryModel> getRepositoryModels() {
        long methodStart = System.currentTimeMillis();
        List<String> list = this.getRepositoryList();
        ArrayList<RepositoryModel> repositories = new ArrayList<RepositoryModel>();
        for (String repo : list) {
            RepositoryModel model = this.getRepositoryModel(repo);
            if (model == null) continue;
            repositories.add(model);
        }
        long duration = System.currentTimeMillis() - methodStart;
        this.logger.info(MessageFormat.format("{0} repository models loaded in {1} msecs", duration));
        return repositories;
    }

    @Override
    public List<RepositoryModel> getRepositoryModels(UserModel user) {
        long methodStart = System.currentTimeMillis();
        List<String> list = this.getRepositoryList();
        ArrayList<RepositoryModel> repositories = new ArrayList<RepositoryModel>();
        for (String repo : list) {
            RepositoryModel model = this.getRepositoryModel(user, repo);
            if (model == null) continue;
            if (!model.hasCommits) {
                if (!UserModel.ANONYMOUS.canPush(model) && (user == null || !user.canPush(model))) continue;
                repositories.add(model);
                continue;
            }
            repositories.add(model);
        }
        long duration = System.currentTimeMillis() - methodStart;
        this.logger.info(MessageFormat.format("{0} repository models loaded for {1} in {2} msecs", repositories.size(), user == null ? "anonymous" : user.username, duration));
        return repositories;
    }

    @Override
    public RepositoryModel getRepositoryModel(UserModel user, String repositoryName) {
        RepositoryModel model = this.getRepositoryModel(repositoryName);
        if (model == null) {
            return null;
        }
        if (user == null) {
            user = UserModel.ANONYMOUS;
        }
        if (user.canView(model)) {
            return model;
        }
        return null;
    }

    @Override
    public RepositoryModel getRepositoryModel(String name) {
        String repositoryName = this.fixRepositoryName(name);
        String repositoryKey = this.getRepositoryKey(repositoryName);
        if (!this.repositoryListCache.containsKey(repositoryKey)) {
            RepositoryModel model = this.loadRepositoryModel(repositoryName);
            if (model == null) {
                return null;
            }
            this.addToCachedRepositoryList(model);
            return DeepCopier.copy(model);
        }
        RepositoryModel model = this.repositoryListCache.get(repositoryKey);
        if (this.isCollectingGarbage(model.name)) {
            RepositoryModel rm = DeepCopier.copy(model);
            rm.isCollectingGarbage = true;
            return rm;
        }
        Repository r = this.getRepository(model.name);
        if (r == null) {
            this.removeFromCachedRepositoryList(repositoryName);
            this.logger.error(MessageFormat.format("Repository \"{0}\" is missing! Removing from cache.", repositoryName));
            return null;
        }
        FileBasedConfig config = (FileBasedConfig)this.getRepositoryConfig(r);
        if (config.isOutdated()) {
            this.logger.debug(MessageFormat.format("Config for \"{0}\" has changed. Reloading model and updating cache.", repositoryName));
            model = this.loadRepositoryModel(model.name);
            this.removeFromCachedRepositoryList(model.name);
            this.addToCachedRepositoryList(model);
        } else {
            if (!model.hasCommits) {
                model.hasCommits = JGitUtils.hasCommits(r);
            }
            this.updateLastChangeFields(r, model);
        }
        r.close();
        return DeepCopier.copy(model);
    }

    @Override
    public long getStarCount(RepositoryModel repository) {
        long count = 0L;
        for (UserModel user : this.userManager.getAllUsers()) {
            if (!user.getPreferences().isStarredRepository(repository.name)) continue;
            ++count;
        }
        return count;
    }

    private String fixRepositoryName(String repositoryName) {
        if (StringUtils.isEmpty(repositoryName)) {
            return repositoryName;
        }
        String name = repositoryName.replace("%7E", "~").replace("%7e", "~");
        if ((name = name.replace("%2F", "/").replace("%2f", "/")).charAt(name.length() - 1) == '/') {
            name = name.substring(0, name.length() - 1);
        }
        char lastChar = '/';
        StringBuilder sb = new StringBuilder();
        for (char c : name.toCharArray()) {
            if (c == '/' && lastChar == c) continue;
            sb.append(c);
            lastChar = c;
        }
        return sb.toString();
    }

    private String getRepositoryKey(String repositoryName) {
        String name = this.fixRepositoryName(repositoryName);
        return StringUtils.stripDotGit(name).toLowerCase();
    }

    private StoredConfig getRepositoryConfig(Repository r) {
        try {
            Field f = r.getClass().getDeclaredField("repoConfig");
            f.setAccessible(true);
            StoredConfig config = (StoredConfig)f.get(r);
            return config;
        }
        catch (Exception e) {
            this.logger.error("Failed to retrieve \"repoConfig\" via reflection", (Throwable)e);
            return r.getConfig();
        }
    }

    private RepositoryModel loadRepositoryModel(String repositoryName) {
        Repository r = this.getRepository(repositoryName);
        if (r == null) {
            return null;
        }
        RepositoryModel model = new RepositoryModel();
        model.isBare = r.isBare();
        File basePath = this.getRepositoriesFolder();
        model.name = model.isBare ? com.gitblit.utils.FileUtils.getRelativePath(basePath, r.getDirectory()) : com.gitblit.utils.FileUtils.getRelativePath(basePath, r.getDirectory().getParentFile());
        if (StringUtils.isEmpty(model.name)) {
            model.name = repositoryName;
        }
        model.projectPath = StringUtils.getFirstPathElement(repositoryName);
        StoredConfig config = r.getConfig();
        boolean hasOrigin = false;
        if (config != null) {
            String desc;
            File descFile;
            boolean bl = hasOrigin = !StringUtils.isEmpty(config.getString("remote", "origin", "url"));
            if (this.getConfig(config, "description", null) == null && (descFile = new File(r.getDirectory(), "description")).exists() && !(desc = com.gitblit.utils.FileUtils.readContent(descFile, System.getProperty("line.separator"))).toLowerCase().startsWith("unnamed repository")) {
                config.setString("gitblit", null, "description", desc);
            }
            model.description = this.getConfig(config, "description", "");
            model.originRepository = this.getConfig(config, "originRepository", null);
            model.addOwners(ArrayUtils.fromString(this.getConfig(config, "owner", "")));
            model.acceptNewPatchsets = this.getConfig(config, "acceptNewPatchsets", true);
            model.acceptNewTickets = this.getConfig(config, "acceptNewTickets", true);
            model.requireApproval = this.getConfig(config, "requireApproval", this.settings.getBoolean("tickets.requireApproval", false));
            model.mergeTo = this.getConfig(config, "mergeTo", null);
            model.mergeType = Constants.MergeType.fromName(this.getConfig(config, "mergeType", this.settings.getString("tickets.mergeType", null)));
            model.useIncrementalPushTags = this.getConfig(config, "useIncrementalPushTags", false);
            model.incrementalPushTagPrefix = this.getConfig(config, "incrementalPushTagPrefix", null);
            model.allowForks = this.getConfig(config, "allowForks", true);
            model.accessRestriction = Constants.AccessRestrictionType.fromName(this.getConfig(config, "accessRestriction", this.settings.getString("git.defaultAccessRestriction", "PUSH")));
            model.authorizationControl = Constants.AuthorizationControl.fromName(this.getConfig(config, "authorizationControl", this.settings.getString("git.defaultAuthorizationControl", null)));
            model.verifyCommitter = this.getConfig(config, "verifyCommitter", false);
            model.showRemoteBranches = this.getConfig(config, "showRemoteBranches", hasOrigin);
            model.isFrozen = this.getConfig(config, "isFrozen", false);
            model.skipSizeCalculation = this.getConfig(config, "skipSizeCalculation", false);
            model.skipSummaryMetrics = this.getConfig(config, "skipSummaryMetrics", false);
            model.commitMessageRenderer = Constants.CommitMessageRenderer.fromName(this.getConfig(config, "commitMessageRenderer", this.settings.getString("web.commitMessageRenderer", null)));
            model.federationStrategy = Constants.FederationStrategy.fromName(this.getConfig(config, "federationStrategy", null));
            model.federationSets = new ArrayList<String>(Arrays.asList(config.getStringList("gitblit", null, "federationSets")));
            model.isFederated = this.getConfig(config, "isFederated", false);
            model.gcThreshold = this.getConfig(config, "gcThreshold", this.settings.getString("git.defaultGarbageCollectionThreshold", "500KB"));
            model.gcPeriod = this.getConfig(config, "gcPeriod", this.settings.getInteger("git.defaultGarbageCollectionPeriod", 7));
            try {
                model.lastGC = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ").parse(this.getConfig(config, "lastGC", "1970-01-01'T'00:00:00Z"));
            }
            catch (Exception e) {
                model.lastGC = new Date(0L);
            }
            model.maxActivityCommits = this.getConfig(config, "maxActivityCommits", this.settings.getInteger("web.maxActivityCommits", 0));
            model.origin = config.getString("remote", "origin", "url");
            if (model.origin != null) {
                model.origin = model.origin.replace('\\', '/');
                model.isMirror = config.getBoolean("remote", "origin", "mirror", false);
            }
            model.preReceiveScripts = new ArrayList<String>(Arrays.asList(config.getStringList("gitblit", null, "preReceiveScript")));
            model.postReceiveScripts = new ArrayList<String>(Arrays.asList(config.getStringList("gitblit", null, "postReceiveScript")));
            model.mailingLists = new ArrayList<String>(Arrays.asList(config.getStringList("gitblit", null, "mailingList")));
            model.indexedBranches = new ArrayList<String>(Arrays.asList(config.getStringList("gitblit", null, "indexBranch")));
            model.metricAuthorExclusions = new ArrayList<String>(Arrays.asList(config.getStringList("gitblit", null, "metricAuthorExclusions")));
            model.customFields = new LinkedHashMap<String, String>();
            for (String aProperty : config.getNames("gitblit", "customFields")) {
                model.customFields.put(aProperty, config.getString("gitblit", "customFields", aProperty));
            }
        }
        model.HEAD = JGitUtils.getHEADRef(r);
        if (StringUtils.isEmpty(model.mergeTo)) {
            model.mergeTo = model.HEAD;
        }
        model.availableRefs = JGitUtils.getAvailableHeadTargets(r);
        model.sparkleshareId = JGitUtils.getSparkleshareId(r);
        model.hasCommits = JGitUtils.hasCommits(r);
        this.updateLastChangeFields(r, model);
        r.close();
        if (StringUtils.isEmpty(model.originRepository) && model.origin != null && model.origin.startsWith("file://")) {
            try {
                File repoFolder;
                File folder = new File(new URI(model.origin));
                String originRepo = com.gitblit.utils.FileUtils.getRelativePath(this.getRepositoriesFolder(), folder);
                if (!StringUtils.isEmpty(originRepo) && (repoFolder = new File(this.getRepositoriesFolder(), originRepo)).exists()) {
                    model.originRepository = originRepo.toLowerCase();
                    this.updateConfiguration(r, model);
                }
            }
            catch (URISyntaxException e) {
                this.logger.error("Failed to determine fork for " + model, (Throwable)e);
            }
        }
        return model;
    }

    @Override
    public boolean hasRepository(String repositoryName) {
        return this.hasRepository(repositoryName, false);
    }

    @Override
    public boolean hasRepository(String repositoryName, boolean caseSensitiveCheck) {
        if (!caseSensitiveCheck && this.settings.getBoolean("git.cacheRepositoryList", true)) {
            String key = this.getRepositoryKey(repositoryName);
            return this.repositoryListCache.containsKey(key);
        }
        Repository r = this.getRepository(repositoryName, false);
        if (r == null) {
            return false;
        }
        r.close();
        return true;
    }

    @Override
    public boolean hasFork(String username, String origin) {
        return this.getFork(username, origin) != null;
    }

    @Override
    public String getFork(String username, String origin) {
        if (StringUtils.isEmpty(origin)) {
            return null;
        }
        String userProject = ModelUtils.getPersonalPath(username);
        if (this.settings.getBoolean("git.cacheRepositoryList", true)) {
            String originKey = this.getRepositoryKey(origin);
            String userPath = userProject + "/";
            HashSet<Object> roots = new HashSet<Object>();
            roots.add(originKey);
            RepositoryModel originModel = this.repositoryListCache.get(originKey);
            while (originModel != null) {
                if (!ArrayUtils.isEmpty(originModel.forks)) {
                    for (String fork : originModel.forks) {
                        if (fork.startsWith(userPath)) continue;
                        roots.add(fork.toLowerCase());
                    }
                }
                if (originModel.originRepository != null) {
                    String ooKey = this.getRepositoryKey(originModel.originRepository);
                    roots.add(ooKey);
                    originModel = this.repositoryListCache.get(ooKey);
                    continue;
                }
                originModel = null;
            }
            for (String repository : this.repositoryListCache.keySet()) {
                String ooKey;
                if (!repository.startsWith(userPath)) continue;
                RepositoryModel model = this.repositoryListCache.get(repository);
                if (StringUtils.isEmpty(model.originRepository) || !roots.contains(ooKey = this.getRepositoryKey(model.originRepository))) continue;
                return model.name;
            }
        } else {
            File subfolder = new File(this.getRepositoriesFolder(), userProject);
            List<String> repositories = JGitUtils.getRepositoryList(subfolder, this.settings.getBoolean("git.onlyAccessBareRepositories", false), this.settings.getBoolean("git.searchRepositoriesSubfolders", true), this.settings.getInteger("git.searchRecursionDepth", -1), this.settings.getStrings("git.searchExclusions"));
            for (String repository : repositories) {
                RepositoryModel model = this.getRepositoryModel(userProject + "/" + repository);
                if (model.originRepository == null || !model.originRepository.equalsIgnoreCase(origin)) continue;
                return model.name;
            }
        }
        return null;
    }

    @Override
    public ForkModel getForkNetwork(String repository) {
        if (this.settings.getBoolean("git.cacheRepositoryList", true)) {
            String key = this.getRepositoryKey(repository);
            RepositoryModel model = this.repositoryListCache.get(key);
            if (model == null) {
                return null;
            }
            while (model.originRepository != null) {
                String originKey = this.getRepositoryKey(model.originRepository);
                model = this.repositoryListCache.get(originKey);
                if (model != null) continue;
                return null;
            }
            ForkModel root = this.getForkModelFromCache(model.name);
            return root;
        }
        RepositoryModel model = this.getRepositoryModel(repository.toLowerCase());
        while (model.originRepository != null) {
            model = this.getRepositoryModel(model.originRepository);
        }
        ForkModel root = this.getForkModel(model.name);
        return root;
    }

    private ForkModel getForkModelFromCache(String repository) {
        String key = this.getRepositoryKey(repository);
        RepositoryModel model = this.repositoryListCache.get(key);
        if (model == null) {
            return null;
        }
        ForkModel fork = new ForkModel(model);
        if (!ArrayUtils.isEmpty(model.forks)) {
            for (String aFork : model.forks) {
                ForkModel fm = this.getForkModelFromCache(aFork);
                if (fm == null) continue;
                fork.forks.add(fm);
            }
        }
        return fork;
    }

    private ForkModel getForkModel(String repository) {
        RepositoryModel model = this.getRepositoryModel(repository.toLowerCase());
        if (model == null) {
            return null;
        }
        ForkModel fork = new ForkModel(model);
        if (!ArrayUtils.isEmpty(model.forks)) {
            for (String aFork : model.forks) {
                ForkModel fm = this.getForkModel(aFork);
                if (fm == null) continue;
                fork.forks.add(fm);
            }
        }
        return fork;
    }

    @Override
    public long updateLastChangeFields(Repository r, RepositoryModel model) {
        JGitUtils.LastChange lc = JGitUtils.getLastChange(r);
        model.lastChange = lc.when;
        model.lastChangeAuthor = lc.who;
        if (!this.settings.getBoolean("web.showRepositorySizes", true) || model.skipSizeCalculation) {
            model.size = null;
            return 0L;
        }
        if (!this.repositorySizeCache.hasCurrent(model.name, model.lastChange)) {
            File gitDir = r.getDirectory();
            long sz = com.gitblit.utils.FileUtils.folderSize(gitDir);
            this.repositorySizeCache.updateObject(model.name, model.lastChange, sz);
        }
        long size = this.repositorySizeCache.getObject(model.name);
        ByteFormat byteFormat = new ByteFormat();
        model.size = byteFormat.format(size);
        return size;
    }

    @Override
    public boolean isIdle(Repository repository) {
        try {
            Field useCnt = Repository.class.getDeclaredField("useCnt");
            useCnt.setAccessible(true);
            int useCount = ((AtomicInteger)useCnt.get(repository)).get();
            return useCount == 2;
        }
        catch (Exception e) {
            this.logger.warn(MessageFormat.format("Failed to reflectively determine use count for repository {0}", repository.getDirectory().getPath()), (Throwable)e);
            return false;
        }
    }

    @Override
    public void closeAll() {
        for (String repository : this.getRepositoryList()) {
            this.close(repository);
        }
    }

    @Override
    public void close(String repositoryName) {
        Repository repository = this.getRepository(repositoryName);
        if (repository == null) {
            return;
        }
        RepositoryCache.close((Repository)repository);
        int uses = 2;
        try {
            Field useCnt = Repository.class.getDeclaredField("useCnt");
            useCnt.setAccessible(true);
            uses = ((AtomicInteger)useCnt.get(repository)).get();
        }
        catch (Exception e) {
            this.logger.warn(MessageFormat.format("Failed to reflectively determine use count for repository {0}", repositoryName), (Throwable)e);
        }
        if (uses > 0) {
            this.logger.debug(MessageFormat.format("{0}.useCnt={1}, calling close() {2} time(s) to close object and ref databases", repositoryName, uses, uses));
            for (int i = 0; i < uses; ++i) {
                repository.close();
            }
        }
        this.luceneExecutor.close(repositoryName);
    }

    @Override
    public List<Metric> getRepositoryDefaultMetrics(RepositoryModel model, Repository repository) {
        if (this.repositoryMetricsCache.hasCurrent(model.name, model.lastChange)) {
            return new ArrayList<Metric>((Collection)this.repositoryMetricsCache.getObject(model.name));
        }
        List<Metric> metrics = MetricUtils.getDateMetrics(repository, null, true, null, this.runtimeManager.getTimezone());
        this.repositoryMetricsCache.updateObject(model.name, model.lastChange, metrics);
        return new ArrayList<Metric>(metrics);
    }

    private String getConfig(StoredConfig config, String field, String defaultValue) {
        String value = config.getString("gitblit", null, field);
        if (StringUtils.isEmpty(value)) {
            return defaultValue;
        }
        return value;
    }

    private boolean getConfig(StoredConfig config, String field, boolean defaultValue) {
        return config.getBoolean("gitblit", field, defaultValue);
    }

    private int getConfig(StoredConfig config, String field, int defaultValue) {
        String value = config.getString("gitblit", null, field);
        if (StringUtils.isEmpty(value)) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(value);
        }
        catch (Exception exception) {
            return defaultValue;
        }
    }

    @Override
    public void updateRepositoryModel(String repositoryName, RepositoryModel repository, boolean isCreate) throws GitBlitException {
        block31: {
            boolean isRename;
            block30: {
                if (this.isCollectingGarbage(repositoryName)) {
                    throw new GitBlitException(MessageFormat.format("sorry, Gitblit is busy collecting garbage in {0}", repositoryName));
                }
                Repository r = null;
                String projectPath = StringUtils.getFirstPathElement(repository.name);
                if (!StringUtils.isEmpty(projectPath) && projectPath.equalsIgnoreCase(this.settings.getString("web.repositoryRootGroupName", "main"))) {
                    repository.name = repository.name.substring(projectPath.length() + 1);
                }
                isRename = false;
                if (isCreate) {
                    if (!repository.name.toLowerCase().endsWith(".git")) {
                        repository.name = repository.name + ".git";
                    }
                    if (this.hasRepository(repository.name)) {
                        throw new GitBlitException(MessageFormat.format("Can not create repository ''{0}'' because it already exists.", repository.name));
                    }
                    this.logger.info("create repository " + repository.name);
                    String shared = this.settings.getString("git.createRepositoriesShared", "FALSE");
                    r = JGitUtils.createRepository(this.repositoriesFolder, repository.name, shared);
                } else {
                    boolean bl = isRename = !repositoryName.equalsIgnoreCase(repository.name);
                    if (isRename) {
                        String originKey;
                        RepositoryModel origin;
                        if (!repository.name.toLowerCase().endsWith(".git")) {
                            repository.name = repository.name + ".git";
                        }
                        if (new File(this.repositoriesFolder, repository.name).exists()) {
                            throw new GitBlitException(MessageFormat.format("Failed to rename ''{0}'' because ''{1}'' already exists.", repositoryName, repository.name));
                        }
                        this.close(repositoryName);
                        File folder = new File(this.repositoriesFolder, repositoryName);
                        File destFolder = new File(this.repositoriesFolder, repository.name);
                        if (destFolder.exists()) {
                            throw new GitBlitException(MessageFormat.format("Can not rename repository ''{0}'' to ''{1}'' because ''{1}'' already exists.", repositoryName, repository.name));
                        }
                        File parentFile = destFolder.getParentFile();
                        if (!parentFile.exists() && !parentFile.mkdirs()) {
                            throw new GitBlitException(MessageFormat.format("Failed to create folder ''{0}''", parentFile.getAbsolutePath()));
                        }
                        if (!folder.renameTo(destFolder)) {
                            throw new GitBlitException(MessageFormat.format("Failed to rename repository ''{0}'' to ''{1}''.", repositoryName, repository.name));
                        }
                        if (!this.userManager.renameRepositoryRole(repositoryName, repository.name)) {
                            throw new GitBlitException(MessageFormat.format("Failed to rename repository permissions ''{0}'' to ''{1}''.", repositoryName, repository.name));
                        }
                        if (!ArrayUtils.isEmpty(repository.forks)) {
                            for (String fork : repository.forks) {
                                Repository rf = this.getRepository(fork);
                                try {
                                    StoredConfig config = rf.getConfig();
                                    String origin2 = config.getString("remote", "origin", "url");
                                    origin2 = origin2.replace(repositoryName, repository.name);
                                    config.setString("remote", "origin", "url", origin2);
                                    config.setString("gitblit", null, "originRepository", repository.name);
                                    config.save();
                                }
                                catch (Exception e) {
                                    this.logger.error("Failed to update repository fork config for " + fork, (Throwable)e);
                                }
                                rf.close();
                            }
                        }
                        if (!StringUtils.isEmpty(repository.originRepository) && (origin = this.repositoryListCache.get(originKey = this.getRepositoryKey(repository.originRepository))) != null && !ArrayUtils.isEmpty(origin.forks)) {
                            origin.forks.remove(repositoryName);
                            origin.forks.add(repository.name);
                        }
                        this.clearRepositoryMetadataCache(repositoryName);
                        repository.resetDisplayName();
                    }
                    this.logger.info("edit repository " + repository.name);
                    r = this.getRepository(repository.name);
                }
                if (r != null) {
                    this.updateConfiguration(r, repository);
                    File descFile = new File(r.getDirectory(), "description");
                    if (repository.description != null) {
                        com.gitblit.utils.FileUtils.writeContent(descFile, repository.description);
                    } else if (descFile.exists() && !descFile.isDirectory()) {
                        descFile.delete();
                    }
                    String currentRef = JGitUtils.getHEADRef(r);
                    if (!StringUtils.isEmpty(repository.HEAD) && !repository.HEAD.equals(currentRef)) {
                        this.logger.info(MessageFormat.format("Relinking {0} HEAD from {1} to {2}", repository.name, currentRef, repository.HEAD));
                        if (JGitUtils.setHEADtoRef(r, repository.HEAD)) {
                            this.clearRepositoryMetadataCache(repository.name);
                        }
                    }
                    JGitUtils.adjustSharedPerm(new File(r.getDirectory().getAbsolutePath(), "config"), this.settings.getString("git.createRepositoriesShared", "FALSE"));
                    JGitUtils.adjustSharedPerm(new File(r.getDirectory().getAbsolutePath(), "HEAD"), this.settings.getString("git.createRepositoriesShared", "FALSE"));
                    r.close();
                }
                this.removeFromCachedRepositoryList(repositoryName);
                this.addToCachedRepositoryList(repository);
                if (!isCreate || this.pluginManager == null) break block30;
                for (RepositoryLifeCycleListener listener : this.pluginManager.getExtensions(RepositoryLifeCycleListener.class)) {
                    try {
                        listener.onCreation(repository);
                    }
                    catch (Throwable t) {
                        this.logger.error(String.format("failed to call plugin onCreation %s", repositoryName), t);
                    }
                }
                break block31;
            }
            if (!isRename || this.pluginManager == null) break block31;
            for (RepositoryLifeCycleListener listener : this.pluginManager.getExtensions(RepositoryLifeCycleListener.class)) {
                try {
                    listener.onRename(repositoryName, repository);
                }
                catch (Throwable t) {
                    this.logger.error(String.format("failed to call plugin onRename %s", repositoryName), t);
                }
            }
        }
    }

    @Override
    public void updateConfiguration(Repository r, RepositoryModel repository) {
        StoredConfig config = r.getConfig();
        config.setString("gitblit", null, "description", repository.description);
        config.setString("gitblit", null, "originRepository", repository.originRepository);
        config.setString("gitblit", null, "owner", ArrayUtils.toString(repository.owners));
        config.setBoolean("gitblit", null, "acceptNewPatchsets", repository.acceptNewPatchsets);
        config.setBoolean("gitblit", null, "acceptNewTickets", repository.acceptNewTickets);
        if (this.settings.getBoolean("tickets.requireApproval", false) == repository.requireApproval) {
            config.unset("gitblit", null, "requireApproval");
        } else {
            config.setBoolean("gitblit", null, "requireApproval", repository.requireApproval);
        }
        if (!StringUtils.isEmpty(repository.mergeTo)) {
            config.setString("gitblit", null, "mergeTo", repository.mergeTo);
        }
        if (repository.mergeType == null || repository.mergeType == Constants.MergeType.fromName(this.settings.getString("tickets.mergeType", null))) {
            config.unset("gitblit", null, "mergeType");
        } else {
            config.setString("gitblit", null, "mergeType", repository.mergeType.name());
        }
        config.setBoolean("gitblit", null, "useIncrementalPushTags", repository.useIncrementalPushTags);
        if (StringUtils.isEmpty(repository.incrementalPushTagPrefix) || repository.incrementalPushTagPrefix.equals(this.settings.getString("git.defaultIncrementalPushTagPrefix", "r"))) {
            config.unset("gitblit", null, "incrementalPushTagPrefix");
        } else {
            config.setString("gitblit", null, "incrementalPushTagPrefix", repository.incrementalPushTagPrefix);
        }
        config.setBoolean("gitblit", null, "allowForks", repository.allowForks);
        config.setString("gitblit", null, "accessRestriction", repository.accessRestriction.name());
        config.setString("gitblit", null, "authorizationControl", repository.authorizationControl.name());
        config.setBoolean("gitblit", null, "verifyCommitter", repository.verifyCommitter);
        config.setBoolean("gitblit", null, "showRemoteBranches", repository.showRemoteBranches);
        config.setBoolean("gitblit", null, "isFrozen", repository.isFrozen);
        config.setBoolean("gitblit", null, "skipSizeCalculation", repository.skipSizeCalculation);
        config.setBoolean("gitblit", null, "skipSummaryMetrics", repository.skipSummaryMetrics);
        config.setString("gitblit", null, "federationStrategy", repository.federationStrategy.name());
        config.setBoolean("gitblit", null, "isFederated", repository.isFederated);
        config.setString("gitblit", null, "gcThreshold", repository.gcThreshold);
        if (repository.gcPeriod == this.settings.getInteger("git.defaultGarbageCollectionPeriod", 7)) {
            config.unset("gitblit", null, "gcPeriod");
        } else {
            config.setInt("gitblit", null, "gcPeriod", repository.gcPeriod);
        }
        if (repository.lastGC != null) {
            config.setString("gitblit", null, "lastGC", new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ").format(repository.lastGC));
        }
        if (repository.maxActivityCommits == this.settings.getInteger("web.maxActivityCommits", 0)) {
            config.unset("gitblit", null, "maxActivityCommits");
        } else {
            config.setInt("gitblit", null, "maxActivityCommits", repository.maxActivityCommits);
        }
        Constants.CommitMessageRenderer defaultRenderer = Constants.CommitMessageRenderer.fromName(this.settings.getString("web.commitMessageRenderer", null));
        if (repository.commitMessageRenderer == null || repository.commitMessageRenderer == defaultRenderer) {
            config.unset("gitblit", null, "commitMessageRenderer");
        } else {
            config.setString("gitblit", null, "commitMessageRenderer", repository.commitMessageRenderer.name());
        }
        this.updateList(config, "federationSets", repository.federationSets);
        this.updateList(config, "preReceiveScript", repository.preReceiveScripts);
        this.updateList(config, "postReceiveScript", repository.postReceiveScripts);
        this.updateList(config, "mailingList", repository.mailingLists);
        this.updateList(config, "indexBranch", repository.indexedBranches);
        this.updateList(config, "metricAuthorExclusions", repository.metricAuthorExclusions);
        if (repository.customFields != null) {
            if (repository.customFields.size() == 0) {
                config.unsetSection("gitblit", "customFields");
            } else {
                for (Map.Entry<String, String> property : repository.customFields.entrySet()) {
                    String key = property.getKey();
                    String value = property.getValue();
                    config.setString("gitblit", "customFields", key, value);
                }
            }
        }
        try {
            config.save();
        }
        catch (IOException e) {
            this.logger.error("Failed to save repository config!", (Throwable)e);
        }
    }

    private void updateList(StoredConfig config, String field, List<String> list) {
        if (list == null) {
            return;
        }
        if (ArrayUtils.isEmpty(list)) {
            config.unset("gitblit", null, field);
        } else {
            config.setStringList("gitblit", null, field, list);
        }
    }

    @Override
    public boolean canDelete(RepositoryModel repository) {
        return this.settings.getBoolean("web.allowDeletingNonEmptyRepositories", true) || !repository.hasCommits;
    }

    @Override
    public boolean deleteRepositoryModel(RepositoryModel model) {
        return this.deleteRepository(model.name);
    }

    @Override
    public boolean deleteRepository(String repositoryName) {
        RepositoryModel repository = this.getRepositoryModel(repositoryName);
        if (!this.canDelete(repository)) {
            this.logger.warn("Attempt to delete {} rejected!", (Object)repositoryName);
            return false;
        }
        try {
            File folder;
            this.close(repositoryName);
            this.clearRepositoryMetadataCache(repositoryName);
            RepositoryModel model = this.removeFromCachedRepositoryList(repositoryName);
            if (model != null && !ArrayUtils.isEmpty(model.forks)) {
                this.resetRepositoryListCache();
            }
            if ((folder = new File(this.repositoriesFolder, repositoryName)).exists() && folder.isDirectory()) {
                FileUtils.delete((File)folder, (int)3);
                if (this.userManager.deleteRepositoryRole(repositoryName)) {
                    this.logger.info(MessageFormat.format("Repository \"{0}\" deleted", repositoryName));
                    if (this.pluginManager != null) {
                        for (RepositoryLifeCycleListener listener : this.pluginManager.getExtensions(RepositoryLifeCycleListener.class)) {
                            try {
                                listener.onDeletion(repository);
                            }
                            catch (Throwable t) {
                                this.logger.error(String.format("failed to call plugin onDeletion %s", repositoryName), t);
                            }
                        }
                    }
                    return true;
                }
            }
        }
        catch (Throwable t) {
            this.logger.error(MessageFormat.format("Failed to delete repository {0}", repositoryName), t);
        }
        return false;
    }

    @Override
    public List<String> getAllScripts() {
        File groovyFolder = this.getHooksFolder();
        File[] files = groovyFolder.listFiles(new FileFilter(){

            @Override
            public boolean accept(File pathname) {
                return pathname.isFile() && pathname.getName().endsWith(".groovy");
            }
        });
        ArrayList<String> scripts = new ArrayList<String>();
        if (files != null) {
            for (File file : files) {
                String script = file.getName().substring(0, file.getName().lastIndexOf(46));
                scripts.add(script);
            }
        }
        return scripts;
    }

    @Override
    public List<String> getPreReceiveScriptsInherited(RepositoryModel repository) {
        LinkedHashSet<String> scripts = new LinkedHashSet<String>();
        for (String script : this.settings.getStrings("groovy.preReceiveScripts")) {
            if (script.endsWith(".groovy")) {
                scripts.add(script.substring(0, script.lastIndexOf(46)));
                continue;
            }
            scripts.add(script);
        }
        if (repository != null) {
            for (String teamname : this.userManager.getTeamNamesForRepositoryRole(repository.name)) {
                TeamModel team = this.userManager.getTeamModel(teamname);
                if (ArrayUtils.isEmpty(team.preReceiveScripts)) continue;
                scripts.addAll(team.preReceiveScripts);
            }
        }
        return new ArrayList<String>(scripts);
    }

    @Override
    public List<String> getPreReceiveScriptsUnused(RepositoryModel repository) {
        TreeSet<String> inherited = new TreeSet<String>(this.getPreReceiveScriptsInherited(repository));
        ArrayList<String> scripts = new ArrayList<String>();
        for (String script : this.getAllScripts()) {
            if (inherited.contains(script)) continue;
            scripts.add(script);
        }
        return scripts;
    }

    @Override
    public List<String> getPostReceiveScriptsInherited(RepositoryModel repository) {
        LinkedHashSet<String> scripts = new LinkedHashSet<String>();
        for (String script : this.settings.getStrings("groovy.postReceiveScripts")) {
            if (script.endsWith(".groovy")) {
                scripts.add(script.substring(0, script.lastIndexOf(46)));
                continue;
            }
            scripts.add(script);
        }
        if (repository != null) {
            for (String teamname : this.userManager.getTeamNamesForRepositoryRole(repository.name)) {
                TeamModel team = this.userManager.getTeamModel(teamname);
                if (ArrayUtils.isEmpty(team.postReceiveScripts)) continue;
                scripts.addAll(team.postReceiveScripts);
            }
        }
        return new ArrayList<String>(scripts);
    }

    @Override
    public List<String> getPostReceiveScriptsUnused(RepositoryModel repository) {
        TreeSet<String> inherited = new TreeSet<String>(this.getPostReceiveScriptsInherited(repository));
        ArrayList<String> scripts = new ArrayList<String>();
        for (String script : this.getAllScripts()) {
            if (inherited.contains(script)) continue;
            scripts.add(script);
        }
        return scripts;
    }

    @Override
    public List<SearchResult> search(String query, int page, int pageSize, List<String> repositories) {
        List<SearchResult> srs = this.luceneExecutor.search(query, page, pageSize, repositories);
        return srs;
    }

    protected void configureLuceneIndexing() {
        this.luceneExecutor = new LuceneService(this.settings, this);
        String frequency = this.settings.getString("web.luceneFrequency", "2 mins");
        int mins = TimeUtils.convertFrequencyToMinutes(frequency, 2);
        this.scheduledExecutor.scheduleAtFixedRate(this.luceneExecutor, 1L, mins, TimeUnit.MINUTES);
        this.logger.info("Lucene will process indexed branches every {} minutes.", (Object)mins);
    }

    protected void configureGarbageCollector() {
        this.gcExecutor = new GarbageCollectorService(this.settings, this);
        if (this.gcExecutor.isReady()) {
            this.logger.info("Garbage Collector (GC) will scan repositories every 24 hours.");
            Calendar c = Calendar.getInstance();
            c.set(11, this.settings.getInteger("git.garbageCollectionHour", 0));
            c.set(12, 0);
            c.set(13, 0);
            c.set(14, 0);
            Date cd = c.getTime();
            Date now = new Date();
            int delay = 0;
            if (cd.before(now)) {
                c.add(5, 1);
                cd = c.getTime();
            }
            delay = (int)((cd.getTime() - now.getTime()) / 60000L);
            String when = delay + " mins";
            if (delay > 60) {
                when = MessageFormat.format("{0,number,0.0} hours", Float.valueOf((float)delay / 60.0f));
            }
            this.logger.info(MessageFormat.format("Next scheculed GC scan is in {0}", when));
            this.scheduledExecutor.scheduleAtFixedRate(this.gcExecutor, delay, 1440L, TimeUnit.MINUTES);
        } else {
            this.logger.info("Garbage Collector (GC) is disabled.");
        }
    }

    protected void configureMirrorExecutor() {
        this.mirrorExecutor = new MirrorService(this.settings, this);
        if (this.mirrorExecutor.isReady()) {
            int mins = TimeUtils.convertFrequencyToMinutes(this.settings.getString("git.mirrorPeriod", "30 mins"), 5);
            int delay = 1;
            this.scheduledExecutor.scheduleAtFixedRate(this.mirrorExecutor, delay, mins, TimeUnit.MINUTES);
            this.logger.info("Mirror service will fetch updates every {} minutes.", (Object)mins);
            this.logger.info("Next scheduled mirror fetch is in {} minutes", (Object)delay);
        } else {
            this.logger.info("Mirror service is disabled.");
        }
    }

    protected void configureJGit() {
        WindowCacheConfig cfg = new WindowCacheConfig();
        cfg.setPackedGitWindowSize(this.settings.getFilesize("git.packedGitWindowSize", cfg.getPackedGitWindowSize()));
        cfg.setPackedGitLimit(this.settings.getFilesize("git.packedGitLimit", cfg.getPackedGitLimit()));
        cfg.setDeltaBaseCacheLimit(this.settings.getFilesize("git.deltaBaseCacheLimit", cfg.getDeltaBaseCacheLimit()));
        cfg.setPackedGitOpenFiles(this.settings.getInteger("git.packedGitOpenFiles", cfg.getPackedGitOpenFiles()));
        cfg.setPackedGitMMAP(this.settings.getBoolean("git.packedGitMmap", cfg.isPackedGitMMAP()));
        try {
            cfg.install();
            this.logger.debug(MessageFormat.format("{0} = {1,number,0}", "git.packedGitWindowSize", cfg.getPackedGitWindowSize()));
            this.logger.debug(MessageFormat.format("{0} = {1,number,0}", "git.packedGitLimit", cfg.getPackedGitLimit()));
            this.logger.debug(MessageFormat.format("{0} = {1,number,0}", "git.deltaBaseCacheLimit", cfg.getDeltaBaseCacheLimit()));
            this.logger.debug(MessageFormat.format("{0} = {1,number,0}", "git.packedGitOpenFiles", cfg.getPackedGitOpenFiles()));
            this.logger.debug(MessageFormat.format("{0} = {1}", "git.packedGitMmap", cfg.isPackedGitMMAP()));
        }
        catch (IllegalArgumentException e) {
            this.logger.error("Failed to configure JGit parameters!", (Throwable)e);
        }
    }

    protected void configureCommitCache() {
        final int daysToCache = this.settings.getInteger("web.activityCacheDays", 14);
        if (daysToCache <= 0) {
            this.logger.info("Commit cache is disabled");
            return;
        }
        this.logger.info(MessageFormat.format("Preparing {0} day commit cache...", daysToCache));
        CommitCache.instance().setCacheDays(daysToCache);
        Thread loader = new Thread(){

            @Override
            public void run() {
                long start = System.nanoTime();
                long repoCount = 0L;
                long commitCount = 0L;
                Date cutoff = CommitCache.instance().getCutoffDate();
                for (String repositoryName : RepositoryManager.this.getRepositoryList()) {
                    RepositoryModel model = RepositoryManager.this.getRepositoryModel(repositoryName);
                    if (model == null || !model.hasCommits || !model.lastChange.after(cutoff)) continue;
                    ++repoCount;
                    Repository repository = RepositoryManager.this.getRepository(repositoryName);
                    for (RefModel ref : JGitUtils.getLocalBranches(repository, true, -1)) {
                        List<RepositoryCommit> commits;
                        if (!ref.getDate().after(cutoff) || (commits = CommitCache.instance().getCommits(repositoryName, repository, ref.getName())).size() <= 0) continue;
                        RepositoryManager.this.logger.info(MessageFormat.format("  cached {0} commits for {1}:{2}", commits.size(), repositoryName, ref.getName()));
                        commitCount += (long)commits.size();
                    }
                    repository.close();
                }
                RepositoryManager.this.logger.info(MessageFormat.format("built {0} day commit cache of {1} commits across {2} repositories in {3} msecs", daysToCache, commitCount, repoCount, TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - start)));
            }
        };
        loader.setName("CommitCacheLoader");
        loader.setDaemon(true);
        loader.start();
    }

    protected void confirmWriteAccess() {
        try {
            if (!this.getRepositoriesFolder().exists()) {
                this.getRepositoriesFolder().mkdirs();
            }
            File file = File.createTempFile(".test-", ".txt", this.getRepositoriesFolder());
            file.delete();
        }
        catch (Exception e) {
            this.logger.error("");
            this.logger.error("#################################################################");
            this.logger.error("Please check filesystem permissions!");
            this.logger.error("FAILED TO WRITE TO REPOSITORIES FOLDER!!", (Throwable)e);
            this.logger.error("#################################################################");
            this.logger.error("");
        }
    }
}

