/*
 * Decompiled with CFR 0.152.
 */
package com.gitblit.service;

import com.gitblit.IStoredSettings;
import com.gitblit.manager.IRepositoryManager;
import com.gitblit.models.RepositoryModel;
import com.gitblit.utils.FileUtils;
import java.util.Calendar;
import java.util.Date;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.eclipse.jgit.api.GarbageCollectCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.lib.Repository;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GarbageCollectorService
implements Runnable {
    private final Logger logger = LoggerFactory.getLogger(GarbageCollectorService.class);
    private final IStoredSettings settings;
    private final IRepositoryManager repositoryManager;
    private AtomicBoolean running = new AtomicBoolean(false);
    private AtomicBoolean forceClose = new AtomicBoolean(false);
    private final Map<String, GCStatus> gcCache = new ConcurrentHashMap<String, GCStatus>();

    public GarbageCollectorService(IStoredSettings settings, IRepositoryManager repositoryManager) {
        this.settings = settings;
        this.repositoryManager = repositoryManager;
    }

    public boolean isReady() {
        return this.settings.getBoolean("git.enableGarbageCollection", false);
    }

    public boolean isRunning() {
        return this.running.get();
    }

    public boolean lock(String repositoryName) {
        return this.setGCStatus(repositoryName, GCStatus.COLLECTING);
    }

    private boolean setGCStatus(String repositoryName, GCStatus status) {
        String key = repositoryName.toLowerCase();
        if (this.gcCache.containsKey(key) && this.gcCache.get(key).exceeds(GCStatus.READY)) {
            return false;
        }
        this.gcCache.put(key, status);
        return true;
    }

    public boolean isCollectingGarbage(String repositoryName) {
        String key = repositoryName.toLowerCase();
        return this.gcCache.containsKey(key) && GCStatus.COLLECTING.equals((Object)this.gcCache.get(key));
    }

    public void releaseLock(String repositoryName) {
        this.gcCache.put(repositoryName.toLowerCase(), GCStatus.READY);
    }

    public void close() {
        this.forceClose.set(true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        if (!this.isReady()) {
            return;
        }
        this.running.set(true);
        Date now = new Date();
        for (String repositoryName : this.repositoryManager.getRepositoryList()) {
            if (this.forceClose.get()) break;
            if (this.isCollectingGarbage(repositoryName)) {
                this.logger.warn("Already collecting garbage from {}?!?", (Object)repositoryName);
                continue;
            }
            boolean garbageCollected = false;
            RepositoryModel model = null;
            Repository repository = null;
            try {
                model = this.repositoryManager.getRepositoryModel(repositoryName);
                repository = this.repositoryManager.getRepository(repositoryName);
                if (repository == null) {
                    this.logger.warn("GCExecutor is missing repository {}?!?", (Object)repositoryName);
                    continue;
                }
                if (!this.repositoryManager.isIdle(repository)) {
                    this.logger.debug("GCExecutor is skipping {} because it is not idle", (Object)repositoryName);
                    continue;
                }
                if (!this.setGCStatus(repositoryName, GCStatus.COLLECTING)) {
                    this.logger.warn("Can not acquire GC lock for {}, skipping", (Object)repositoryName);
                    continue;
                }
                this.logger.debug("GCExecutor locked idle repository {}", (Object)repositoryName);
                Git git2 = new Git(repository);
                GarbageCollectCommand gc = git2.gc();
                Properties stats = gc.getStatistics();
                Calendar cal = Calendar.getInstance();
                cal.setTime(model.lastGC);
                cal.set(11, 0);
                cal.set(12, 0);
                cal.set(13, 0);
                cal.set(14, 0);
                cal.add(5, model.gcPeriod);
                Date gcDate = cal.getTime();
                boolean shouldCollectGarbage = now.after(gcDate);
                long gcThreshold = FileUtils.convertSizeToLong(model.gcThreshold, 512000L);
                long sizeOfLooseObjects = (Long)stats.get("sizeOfLooseObjects");
                boolean hasEnoughGarbage = sizeOfLooseObjects >= gcThreshold;
                boolean hasGarbage = sizeOfLooseObjects > 0L;
                if (!hasGarbage || !hasEnoughGarbage && !shouldCollectGarbage) continue;
                long looseKB = sizeOfLooseObjects / 1024L;
                this.logger.info("Collecting {} KB of loose objects from {}", (Object)looseKB, (Object)repositoryName);
                gc.call();
                garbageCollected = true;
            }
            catch (Exception e) {
                this.logger.error("Error collecting garbage in {}", (Object)repositoryName, (Object)e);
            }
            finally {
                if (repository != null) {
                    if (garbageCollected) {
                        model.lastGC = new Date();
                        this.repositoryManager.updateConfiguration(repository, model);
                    }
                    repository.close();
                }
                this.releaseLock(repositoryName);
                this.logger.debug("GCExecutor released GC lock for {}", (Object)repositoryName);
            }
        }
        this.running.set(false);
    }

    public static enum GCStatus {
        READY,
        COLLECTING;


        public boolean exceeds(GCStatus s) {
            return this.ordinal() > s.ordinal();
        }
    }
}

